<?php

namespace ISPManager\Clients\Forms;

use Botble\Base\Forms\FieldOptions\NameFieldOption;
use Botble\Base\Forms\FieldOptions\StatusFieldOption;
use Botble\Base\Forms\FieldOptions\OnOffFieldOption;
use Botble\Base\Forms\FieldOptions\TextFieldOption;
use Botble\Base\Forms\FieldOptions\SelectFieldOption;
use Botble\Base\Forms\Fields\SelectField;
use Botble\Base\Forms\Fields\OnOffCheckboxField;
use Botble\Base\Forms\Fields\OnOffField;
use Botble\Base\Forms\Fields\TextField;
use Botble\Base\Forms\Fields\PhoneNumberField;
use Botble\Base\Forms\FormAbstract;
use ISPManager\Clients\Http\Requests\ContactsRequest;
use ISPManager\Clients\Models\Contact;
use ISPManager\Clients\Models\Client;
use ISPManager\Clients\Enums\ClientStatusEnum;

class ContactsForm extends FormAbstract
{
    public function setup(): void
    {
        $clients = Client::query()->pluck('name', 'id')->toArray();
        
        // Get current contact if editing
        $currentContact = $this->getModel();
        $hasPrimaryContact = false;
        $isCurrentPrimary = false;
        
        if ($currentContact) {
            $isCurrentPrimary = $currentContact->is_primary;
            if ($currentContact->client_id) {
                $hasPrimaryContact = Contact::where('client_id', $currentContact->client_id)
                    ->where('is_primary', true)
                    ->where('id', '!=', $currentContact->id)
                    ->exists();
            }
        }

        $this
            ->model(Contact::class)
            ->setValidatorClass(ContactsRequest::class)
            ->columns()
            ->add(
                'client_id',
                SelectField::class,
                SelectFieldOption::make()
                    ->label(trans('plugins/clients::clients.client'))
                    ->required()
                    ->choices(function () {
                        return [null => trans('plugins/clients::contacts.none')] + Client::pluck('name', 'id')->toArray();
                    })
                    ->searchable()
                    ->colspan(2)
            )
            ->add('firstname', TextField::class, 
                    NameFieldOption::make()
                                    ->placeholder(trans('plugins/clients::contacts.firstname'))
                                    ->maxLength(100)
                                    ->label(trans('plugins/clients::contacts.firstname'))
                                    ->required()
                                    ->colspan(1)
            )
            ->add('middlename', TextField::class, 
                NameFieldOption::make()
                                ->placeholder(trans('plugins/clients::contacts.middlename'))
                                ->maxLength(100)
                                ->label(trans('plugins/clients::contacts.middlename'))
                                ->colspan(1)
            )
            ->add('lastname', TextField::class, 
                NameFieldOption::make()
                                ->placeholder(trans('plugins/clients::contacts.lastname'))
                                ->maxLength(100)
                                ->label(trans('plugins/clients::contacts.lastname'))
                                ->required()
                                ->colspan(1)
            )
            ->add('secondlastname', TextField::class, 
                NameFieldOption::make()
                                ->placeholder(trans('plugins/clients::contacts.secondlastname'))
                                ->maxLength(100)
                                ->label(trans('plugins/clients::contacts.secondlastname'))
                                ->colspan(1)
            )
            ->add('nid', TextField::class, 
                NameFieldOption::make()
                                ->placeholder(trans('plugins/clients::contacts.nid'))
                                ->maxLength(20)
                                ->label(trans('plugins/clients::contacts.nid'))
                                ->required()
                                ->colspan(1)
            )
            ->add('email', TextField::class, [
                'label' => trans('plugins/clients::contacts.email'),
                'attr' => ['placeholder' => trans('plugins/clients::contacts.email')],
                'rules' => 'required|email|max:200|unique:contacts,email',
                'colspan' => 1
            ])
            ->add('username', TextField::class, [
                'label' => trans('plugins/clients::contacts.username'),
                'attr' => ['placeholder' => trans('plugins/clients::contacts.username')],
                'rules' => 'required|string|max:50|unique:contacts,username',
                'colspan' => 2
            ])
            ->add(
                'is_change_password',
                OnOffField::class,
                OnOffFieldOption::make()
                    ->label(trans('plugins/clients::contacts.change_password'))
                    ->defaultValue($this->getModel()->exists ? 0 : 1)
                    ->colspan(2)
            )
            ->add(
                'password',
                'password',
                TextFieldOption::make()
                    ->label(trans('plugins/clients::contacts.password'))
                    ->collapsible('is_change_password', 1, ! $this->getModel()->exists || $this->getModel()->is_change_password)
                    ->required(! $this->getModel()->exists)
                    ->maxLength(60)
                    ->colspan(1)
            )
            ->add(
                'password_confirmation',
                'password',
                TextFieldOption::make()
                    ->label(trans('plugins/clients::contacts.password_confirmation'))
                    ->collapsible('is_change_password', 1, ! $this->getModel()->exists || $this->getModel()->is_change_password)
                    ->required(! $this->getModel()->exists)
                    ->maxLength(60)
                    ->colspan(1)
            )
            ->add('phone1', PhoneNumberField::class, 
            NameFieldOption::make()
                            ->placeholder(trans('plugins/clients::contacts.phone'))
                            ->maxLength(25)
                            ->label(trans('plugins/clients::contacts.phone'))
                            ->colspan(1)
            )
            ->add('phone2', PhoneNumberField::class, 
                NameFieldOption::make()
                                ->placeholder(trans('plugins/clients::contacts.phone') . ' #2')
                                ->maxLength(25)
                                ->label(trans('plugins/clients::contacts.phone') . ' #2')
                                ->colspan(1)
            )
            ->add('notify_sms', OnOffCheckboxField::class, [
                'label' => trans('plugins/clients::contacts.sms'),
                'colspan' => 1
            ])
            ->add('notify_whatsapp', OnOffCheckboxField::class, [
                'label' => trans('plugins/clients::contacts.whatsapp'),
                'colspan' => 1
            ])
            ->add('is_primary', OnOffCheckboxField::class, [
                'label' => trans('plugins/clients::contacts.isprimary'),
                'attr' => [
                    'disabled' => $hasPrimaryContact && !$isCurrentPrimary,
                ],
                'helper' => $hasPrimaryContact && !$isCurrentPrimary 
                    ? trans('plugins/clients::contacts.primary_contact_exists') 
                    : null,
                'colspan' => 1
            ])
            ->add('status', SelectField::class, StatusFieldOption::make()
                ->choices(ClientStatusEnum::labels())
                ->colspan(1))
            ->setBreakFieldPoint('status');
    }
}
