<?php

namespace ISPManager\Clients\Tables;

use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Actions\DeleteAction;
use Botble\Table\Actions\EditAction;
use Botble\Table\BulkActions\DeleteBulkAction;
use Botble\Table\BulkChanges\CreatedAtBulkChange;
use Botble\Table\BulkChanges\NameBulkChange;
use Botble\Table\Columns\IdColumn;
use Botble\Table\Columns\NameColumn;
use Botble\Table\Columns\StatusColumn;
use Botble\Table\Columns\Column;
use Botble\Table\HeaderActions\CreateHeaderAction;
use Botble\Table\Columns\Concerns\IconPosition;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use ISPManager\Clients\Tables\BulkChanges\ClientStatusBulkChange;
use ISPManager\Clients\Tables\Columns\CountColumn;
use ISPManager\Clients\Models\Client;

class ClientsTable extends TableAbstract
{
    public function setup(): void
    {
        $this
            ->model(Client::class)
            ->addHeaderAction(CreateHeaderAction::make()->route('clients.create'))
            ->addActions([
                EditAction::make()->route('clients.edit'),
                DeleteAction::make()->route('clients.destroy'),
            ])
            ->addColumns([
                IdColumn::make()
                    ->title(trans('plugins/clients::clients.id')),
                NameColumn::make()->route('clients.contacts.index', ['client' => 'clients.id'])
                    ->title(trans('plugins/clients::clients.client')),
                Column::make('primary_contact_name')
                    ->title(trans('plugins/clients::contacts.primarycontact'))
                    ->alignLeft()
                    ->searchable(true)
                    ->orderable(true),
                CountColumn::make('contacts_count')
                    ->title(trans('plugins/clients::contacts.contacts'))
                    ->icon('ti ti-users')
                    ->iconPosition(IconPosition::End)
                    ->searchable(false)
                    ->orderable(false),
                Column::make('country.name')
                    ->title(trans('plugins/clients::clients.country')),
                StatusColumn::make(),
            ])
            ->addBulkActions([
                DeleteBulkAction::make()->permission('clients.destroy'),
            ])
            ->addBulkChanges([
                NameBulkChange::make(),
                ClientStatusBulkChange::make(),
            ])
            ->queryUsing(function (Builder $query) {
                $query->select([
                        'clients.id',
                        'clients.name',
                        'clients.status',
                        'clients.country_id',
                        DB::raw("CONCAT(contacts.firstname, ' ', contacts.lastname) AS primary_contact_name"),
                    ])
                    ->leftJoin('contacts', function ($join) {
                        $join->on('contacts.client_id', '=', 'clients.id')
                             ->where('contacts.is_primary', 1);  // Only Main contact
                    })
                    ->leftJoin('countries', 'clients.country_id', '=', 'countries.id')
                    ->with('country')
                    ->withCount('contacts');
            })
            ->onAjax(function ($table) {
                return $table->toJson(
                    $table->table
                        ->eloquent($table->query())
                        ->filter(function ($query) {
                            if ($keyword = $this->request->input('search.value')) {
                                $keyword = '%' . strtolower($keyword) . '%';
                                
                                // Filter by the other columns
                                $query->where(function ($query) use ($keyword) {
                                    $query->orWhere('clients.id', 'LIKE', $keyword)
                                          ->orWhere('clients.name', 'LIKE', $keyword)
                                          ->orWhere('clients.status', 'LIKE', $keyword)
                                          ->orWhere('countries.name', 'LIKE', $keyword);
                                });

                                // Mapping with translation for status (English to Spanish)
                                $query->orWhereRaw("LOWER(CASE
                                        WHEN clients.status = 'enabled' THEN '" . trans('plugins/clients::clients.statuses.enabled') . "'
                                        WHEN clients.status = 'disabled' THEN '" . trans('plugins/clients::clients.statuses.disabled') . "'
                                        ELSE clients.status
                                    END) LIKE ?", [$keyword]);
    
                                // Manual filter on the concatenated field primary_contact_name
                                $query->orWhereRaw("LOWER(CONCAT(contacts.firstname, ' ', contacts.lastname)) LIKE ?", [$keyword]);
                            }
                            return $query;
                        })
                );
            });
    }
}
