<?php

namespace ISPManager\Clients\Tables\Columns;

use Botble\Table\Columns\LinkableColumn;
use Botble\Table\Contracts\FormattedColumn as FormattedColumnContract;
use Botble\Base\Contracts\BaseModel;
use Closure;

/**
 * Class ClientNameColumn
 * Custom column for displaying the client's name as a clickable link
 * Extends LinkableColumn to generate a dynamic link based on the client ID
 */
class ClientNameColumn extends LinkableColumn implements FormattedColumnContract
{
    /**
     * Method to initialize the column and define its properties.
     *
     * @param  array|string  $data  The field or data for the column (client.name)
     * @param  string  $name  The name or title of the column
     * @return static
     */
    public static function make(array|string $data = [], string $name = ''): static
    {
        return parent::make($data ?: 'client.name', $name) // Default to 'client.name' field
            ->title(trans('plugins/clients::clients.client'))  // Column title (translate it as needed)
            ->alignStart();  // Align the text to the start (left)
    }

    /**
     * Generate the URL for the link based on the client ID.
     *
     * @param  mixed  $value  The value to be used in the URL (client.name)
     * @return string|null  The URL or null if invalid
     */
    public function getUrl($value): ?string
    {
        // Get the model instance for the item
        $item = $this->getItem();

        if (!$item instanceof BaseModel) {
            return null;
        }

        // Get the client ID
        $clientId = $item->client_id; // Ensure that client_id is available in the query

        // Generate and return the URL for the client's edit page
        return route('clients.edit', ['clients' => $clientId]); // Use the client ID in the URL
    }

    /**
     * Format the value as a link, making the client name clickable.
     *
     * @param  mixed  $value  The value to be displayed as a clickable link (client name)
     * @return string  The HTML link to be rendered
     */
    public function formattedValue($value): ?string
    {
        $item = $this->getItem();

        // Fallback to client name if $value is empty
        $displayValue = $value;
        if (empty($displayValue) && isset($item->client) && isset($item->client->name)) {
            $displayValue = $item->client->name;
        }
        if (empty($displayValue)) {
            $displayValue = $this->getOriginalValue();
        }

        $url = $this->getUrl($value);
        if (!$url) {
            return $displayValue;
        }

        return \Botble\Base\Facades\Html::link($url, $displayValue, ['title' => $displayValue])->toHtml();
    }
}
